/******************************************************************************
 * File     :  t_temp.c
 * Purpose  :  Temperature DS1820 1-Wire Reading (0-127C), External power
 * Author   :  Prajin Palangsantikul
 * E-mail   :  prajin@appsofttech.com
 * Release  :  PCW Compiler 3.180
 * Copyright (c) 2004 APPSOFTTECH CO.,LTD.
 *****************************************************************************/

#define _PIC16F877_           // Use PIC16F877 device

#ifdef _PIC16F877_
#include <16F877.h>           // Standard Header file for the PIC16F877 device
#define  TxD         PIN_C6   // Define Transmitted Data
#define  RxD         PIN_C7   // Define Received Data
#define  CLOCK_SP    20000000 // Clock Speed(Hz)

#define  MAX_DATA_EEPROM   256

#else
#include <16F628.h>           // Standard Header file for the PIC16F628 device
#define  TxD         PIN_A2   // Define Transmitted Data
#define  RxD         PIN_A3   // Define Received Data
#define  CLOCK_SP    4000000  // Clock Speed(Hz)

#define  MAX_DATA_EEPROM   128

#endif

// Device Specification
#fuses HS                     // Oscillator mode HS
#fuses NOLVP, NOWDT           // No Low Voltage Program, No Watchdog timer
#fuses NOPROTECT              // Code no protection
#use delay (clock=CLOCK_SP)   // Use built-in function: delay_ms() & delay_us()
#use rs232(baud=9600, xmit=TxD,rcv=RxD) // Use serial I/O port (RS232)

/***********************************************************************
 *	Includes
 ***********************************************************************/
#include <touch.c>


/***********************************************************************
 *	Constants
 ***********************************************************************/
#define STX 0x02        // Start of text
#define ETX 0x03        // End of text
#define EOT 0x04        // End of transmission

#define ACK 0x06        // Acknowledge
#define NAK 0x15        // Negative acknowledgement
#define ETB 0x17        // End of transmission block
#define FS  0x1C        // Field separator
#define GS  0x1D        // Group separator
#define RS  0x1E        // Record separator
#define CR  0xHD        // Carriage return
#define LF  0xHA        // Carriage return

#define TEMP_START         0x20
#define PROCESS_STOP       0x21
#define EEPROM_READ        0x22
#define EEPROM_WRITE       0x23
#define EEPROM_STOP        0x24
#define INIT_STATUS        0x25
#define TIME_SET_S         0x26
#define TIME_SET_M         0x27
#define TIME_NOTUSE        0x28
#define TIME_SETTRIGGER    0x29
#define IO_SET_CONTROL     0x30

// 1-Wire connect to PIN B0
// I/O Control connect to
// RA0, RA1, RA2, RA3, RA4
// RB3, RB4, RB5, RB6, RB7


/***********************************************************************
 *	Global variables
 ***********************************************************************/
int buff_serial[10];
int sw_task;
// address read&write eeprom
int addr_bit_sign=0;    // count sign bit
int addr_temp=1;        // count address write

int w_eeprom;           // true=wirte, false=stop
int time_temp_sensor;   // wait for read temp sensor
int time_use;           // 1=sec, 2=min
int temp_TH, temp_TL;   // TH, TL Temperature Trigger

//#define HYPER_TERMINAL
// protocol  start->STX , process->buff_serial[0] , data->buff_serial[n] , stop->ETX

/***********************************************************************
 * FUNCTION:     serial_isr
 * DESCRIPTION:  Serial Interrupt
 * PARAMETERS:   nothing
 * RETURNED:     nothing
 ***********************************************************************/
#int_rda
void serial_isr() {

   char ch;
   static int i;

   ch = getc();

   switch(ch) {
   case STX:
      i = 0; sw_task = 0;
   break;

   case ETX:
      sw_task = buff_serial[0];
   break;

   default:
      buff_serial[i++] = ch;
   }
}


/***********************************************************************
 * FUNCTION:     delay_seconds
 * DESCRIPTION:  Delay Seconds
 * PARAMETERS:   n
 * RETURNED:     nothing
 ***********************************************************************/
void delay_seconds(int n) {
  for (;n!=0; n--)
  delay_ms( 1000 );
}


/***********************************************************************
 * FUNCTION:     delay_minute
 * DESCRIPTION:  Delay Minute
 * PARAMETERS:   n
 * RETURNED:     nothing
 ***********************************************************************/
void delay_minute(int n) {
   for (;n!=0; n--)
   delay_seconds(58);
}


/***********************************************************************
 * FUNCTION:     SetTrigger
 * DESCRIPTION:  Set Trigger
 * PARAMETERS:   temp_tl, temp_th
 * RETURNED:     nothing
 ***********************************************************************/
void SetTrigger(byte temp_tl,byte temp_th) {

   if(touch_present()) {
//      printf("\r\nHooK Set trigger %x,%x",temp_th,temp_tl);
      touch_write_byte(0xCC);       // skip rom
      touch_write_byte(0x4E);
      touch_write_byte (temp_th);   // TH
      touch_write_byte (temp_tl);   // TL
      output_high(TOUCH_PIN);
      touch_present();
   }
   sw_task = TEMP_START;            // Temperature task
}


/***********************************************************************
 * FUNCTION:     ReadDataLogger
 * DESCRIPTION:  Read DataLogger
 * PARAMETERS:   nothing
 * RETURNED:     nothing
 ***********************************************************************/
void ReadDataLogger(void) {

   int volume;
   byte buffer;

   putc('E');
   for(volume=0; volume<127; ++volume) {
      buffer = read_EEPROM (volume);
      printf ("%x", buffer);
   }
   putc('S');
   sw_task = TEMP_START;         // Temperature task
}


/***********************************************************************
 * FUNCTION:     WriteDataLogger
 * DESCRIPTION:  Write DataLogger
 * PARAMETERS:   buf0, buf1
 * RETURNED:     nothing
 ***********************************************************************/
void WriteDataLogger(byte buf0, byte buf1) {

   byte buffer;

   buffer = buf0;
   rotate_right(&buffer, 1);
   write_eeprom(addr_bit_sign+addr_temp,buffer);   // wirte temperature

   // write sign bit temperature
   if(buf1) {
      buffer = read_EEPROM (addr_bit_sign);
      shift_left(&buffer, 1, 1);
      write_eeprom(addr_bit_sign,buffer);
   } else {
      buffer = read_EEPROM (addr_bit_sign);
      shift_left(&buffer, 1, 0);
      write_eeprom(addr_bit_sign,buffer);
   }

   if(addr_temp++ >= 8) {
      addr_bit_sign = addr_bit_sign + 9;
      addr_temp = 1;
      if(addr_bit_sign >= 126) addr_bit_sign = 0;
   }
}


/***********************************************************************
 * FUNCTION:     ReadTempDS1820
 * DESCRIPTION:  Temperature DS1820 1-Wire
 * PARAMETERS:   nothing
 * RETURNED:     nothing
 ***********************************************************************/
void ReadTempDS1820(void) {

   byte i;
   byte buffer[9];
   byte temp_msb,temp_lsb,temp_half;

   switch(time_use) {
   case 1:
      delay_seconds(time_temp_sensor);
   break;
   case 2:
      delay_minute(time_temp_sensor);
   break;
   }

   if(touch_present()) {
      touch_write_byte(0xCC);
      touch_write_byte (0x44);
      output_high(TOUCH_PIN);
      delay_ms(2000);
      touch_present();
      touch_write_byte(0xCC);
      touch_write_byte (0xBE);
      for(i=0; i<9;i++)
         buffer[i] = touch_read_byte();

      temp_msb = buffer[1];
      temp_lsb = buffer[0];
      temp_TH  = buffer[2];
      temp_TL  = buffer[3];

      temp_half = 0;
      if(temp_lsb & 0x01) temp_half = 5;
      if(temp_msb <=0x80) temp_lsb = (temp_lsb/2);
      temp_msb = temp_msb & 0x80;
      if(temp_msb >=0x80) {
         temp_lsb = (~temp_lsb)+1;
         temp_lsb = (temp_lsb/2);
         temp_lsb = (-1)*temp_lsb;
      }

#ifdef HYPER_TERMINAL
      printf ("Temperature: %c%3.1f C\r\n", (buffer[1])?'-':' ', (float)buffer[0]/2);
#else
      putc('T');
//      printf ("%c%3.1f", (buffer[1])?'-':' ', (float)buffer[0]/2);
      printf ("%c%d.%d", (buffer[1])?'-':' ', temp_lsb,temp_half);
      putc('S');
#endif
     }
     // write temperature to eeprom
     if(w_eeprom) WriteDataLogger(buffer[0],buffer[1]);
}


/***********************************************************************
 * FUNCTION:     StatusProcess
 * DESCRIPTION:  Status Process
 * PARAMETERS:   nothing
 * RETURNED:     nothing
 ***********************************************************************/
void StatusProcess(void) {

   putc('I');
   printf ("%d%X%X", w_eeprom,temp_TH, temp_TL);
   putc('S');
   sw_task = TEMP_START;         // Temperature task
}


/***********************************************************************
 * FUNCTION:     SetIO_ControlON
 * DESCRIPTION:  Control I/O port
 * PARAMETERS:   nothing
 * RETURNED:     nothing
 ***********************************************************************/
void SetIO_ControlON(unsigned char status_bit) {

   output_bit(PIN_A0, status_bit);
   output_bit(PIN_A1, status_bit);
   output_bit(PIN_A2, status_bit);
   output_bit(PIN_A3, status_bit);
   output_bit(PIN_A5, status_bit);
   output_bit(PIN_B3, status_bit);
   output_bit(PIN_B4, status_bit);
   output_bit(PIN_B5, status_bit);
   output_bit(PIN_B6, status_bit);
   output_bit(PIN_B7, status_bit);
}


/***********************************************************************
 * FUNCTION:     SetIO_Control
 * DESCRIPTION:  Control I/O port
 * PARAMETERS:   bit_p, status_bit
 * RETURNED:     nothing
 ***********************************************************************/
void SetIO_Control(int bit_p, unsigned char status_bit) {

   printf ("I/O Control: %d%d \r\n", bit_p, status_bit);
   switch(bit_p) {
   case 10: output_bit(PIN_A0, status_bit);
      break;
   case 11: output_bit(PIN_A1, status_bit);
      break;
   case 12: output_bit(PIN_A2, status_bit);
      break;
   case 13: output_bit(PIN_A3, status_bit);
      break;
   case 14: output_bit(PIN_A5, status_bit);
      break;
   case 15: output_bit(PIN_B3, status_bit);
      break;
   case 16: output_bit(PIN_B4, status_bit);
      break;
   case 17: output_bit(PIN_B5, status_bit);
      break;
   case 18: output_bit(PIN_B6, status_bit);
      break;
   case 19: output_bit(PIN_B7, status_bit);
      break;
   case 20: SetIO_ControlON(status_bit);
      break;
   }

   sw_task = TEMP_START;         // Temperature task
}


/***********************************************************************
 * FUNCTION:     ShowBuffer
 * DESCRIPTION:  Show Buffer data
 * PARAMETERS:   nothing
 * RETURNED:     nothing
 ***********************************************************************/
void ShowBuffer(void) {

   int i;

   printf("Buffer[");
   for(i=0; i<10; i++)
      printf("%d, ",buff_serial[i]);
   printf("]");
}


/***********************************************************************
 * FUNCTION:    Main
 * DESCRIPTION: This is the main entry point for the program.
 * PARAMETERS:  nothing
 * RETURNED:    nothing
 ***********************************************************************/
void main(void) {

   enable_interrupts(GLOBAL);
   enable_interrupts(int_rda);

   sw_task = TEMP_START;   // Temperature task
   w_eeprom = FALSE;       // Not write temp to eeprom
   time_use = TRUE;
   time_temp_sensor = 0;

   while(TRUE) {
      switch(sw_task) {
      case TEMP_START:  // Read Temperature
         ReadTempDS1820();
      break;

      case EEPROM_READ:  // Read Data Logger from EEPROM internal
         ReadDataLogger();
      break;

      case INIT_STATUS:  // status process in mcu
         StatusProcess();
      break;

      case EEPROM_WRITE:
         w_eeprom = TRUE;     // start write data logger
      break;

      case EEPROM_STOP:
         w_eeprom = FALSE;     // stop write data logger
      break;

      case TIME_SETTRIGGER:  // set time trigger
         SetTrigger(buff_serial[1],buff_serial[2]);
      break;

      case TIME_SET_S:
         time_temp_sensor = (int) buff_serial[1];
         time_use = 1;     // use time seconds
      break;

      case TIME_SET_M:
         time_temp_sensor = (int) buff_serial[1];
         time_use = 2;
      break;

      case TIME_NOTUSE:
         time_use = FALSE;
      break;

      case IO_SET_CONTROL:  // I/O control
         ShowBuffer();
         SetIO_Control(buff_serial[1],buff_serial[2]);
      break;

      case PROCESS_STOP:
         sw_task = 0;
      break;
      default:
      }
   }
}
